# EXPORT : functions ending by export are called from xml
# CRON : functions ending by cron are called from timer
# SCHEDULE : functions ending by schedule are called from cron



# ======
# ENGINE
# ======

Engine = {};

Engine.new = func {
   var obj = { parents : [Engine],

           enginecontrol : EngineControl.new(),
           airdoor : AirDoor.new(),
           bucket : Bucket.new(),
           intake : Intake.new(),
           rating : Rating.new(),
           throttles : EngineThrottle.new(),

           engines : nil,
           engcontrols : nil,

           CUTOFFSEC : 1.0,

           OILPSI : 15.0
         };

   obj.init();

   return obj;
};

Engine.init = func {
   me.engines = props.globals.getNode("/engines").getChildren("engine");
   me.engcontrols = props.globals.getNode("/controls/engines").getChildren("engine");
}

Engine.amber_intake = func( index ) {
    return me.intake.amber_intake( index );
}

Engine.red_engine = func( index ) {
    var result = constant.TRUE;

    if( me.engines[index].getChild("oil-pressure-psi").getValue() > me.OILPSI ) {
        result = constant.FALSE;
    }

    return result;
}

Engine.red_throttle = func {
    return me.throttles.red_throttle();
}

Engine.set_rate = func( rates ) {
    me.bucket.set_rate( rates );
}

Engine.set_throttle = func( position ) {
    return me.rating.set_throttle( position );
}

# simplified engine start (2D panel)
Engine.cutoffexport = func {
   # delay for starter
   settimer(func { me.cutoffcron(); }, me.CUTOFFSEC);
}

Engine.laneexport = func {
    me.intake.laneexport();
}

Engine.schedule = func {
    me.throttles.schedule();
    me.bucket.schedule();
    me.airdoor.schedule();
    me.rating.schedule();
    me.enginecontrol.schedule();
}

Engine.slowschedule = func {
    me.intake.schedule();
}

Engine.cutoffcron = func {
   for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
       # engine start by user
       if( me.engines[i].getChild("starter").getValue() ) {
           if( me.engcontrols[i].getChild("cutoff").getValue() ) {
               me.engcontrols[i].getChild("cutoff").setValue(constant.FALSE);
           }
       }
   }
}


# ===============
# ENGINE THROTTLE
# ===============

EngineThrottle = {};

EngineThrottle.new = func {
   var obj = { parents : [EngineThrottle,System],

               texpath : "Textures/"
             };

   obj.init();

   return obj;
}

EngineThrottle.init = func {
    me.init_ancestor("/systems/engines");
}

EngineThrottle.red_throttle = func {
    var result = constant.FALSE;

    for( var i = 0; i < constantaero.NBENGINES; i = i+1 ) {
         if( me.slave["throttle"][i].getChild("failure").getValue() ) {
             result = constant.TRUE;
             break;
         }
    }

    return result;
}

EngineThrottle.schedule = func {
    var result = constant.FALSE;

    for( var i = 0; i < constantaero.NBENGINES; i = i+1 ) {
         if( !me.slave["throttle"][i].getChild("serviceable").getValue() or
             me.slave["throttle-ctrl"][i].getChild("off").getValue() ) {
             result = constant.FALSE;
         }
         else {
             result = constant.TRUE;
         }

         me.slave["throttle"][i].getChild("available").setValue( result );

         if( !me.slave["throttle"][i].getChild("serviceable").getValue() ) {
             result = constant.TRUE;
         }
         else {
             result = constant.FALSE;
         }

         if( me.slave["throttle"][i].getChild("failure").getValue() != result ) {
             me.slave["throttle"][i].getChild("failure").setValue( result );

             me.failuretexture( i, result );
         }
    }
}

EngineThrottle.failuretexture = func( index, failure ) {
    var texture = "concorde-red-off-throt.rgb";

    if( failure ) {
        texture = "concorde-red-throt.rgb";
    }

    texture = me.texpath ~ texture;

    me.slave["throttle"][index].getChild("failure-texture").setValue( texture );
}


# =======================
# ENGINE CONTROL SCHEDULE
# =======================

EngineControl = {};

EngineControl.new = func {
   var obj = { parents : [EngineControl,System],

           rating : Rating.new(),

           engcontrols : nil,
           engines : nil,
           thecontrol : nil,
           theengines : nil,

           LOWKT : 220,

           speekt : 0,

           HIGHMACH : 1.0,

           speedmach : 0,

           wow : constant.FALSE,
           takeoff : constant.FALSE,
           climb : constant.FALSE,
           reaheat : constant.FALSE,

           FLYOVER : "flyover",
           HIGH : "high",
           MID : "mid",
           LOW : "low",

           value : ""
         };

   obj.init();

   return obj;
}

EngineControl.init = func {
    me.init_ancestor("/systems/engines");

    me.engines = props.globals.getNode("/engines").getChildren("engine");
    me.engcontrols = props.globals.getNode("/controls/engines").getChildren("engine");
    me.thecontrol = props.globals.getNode("/controls/engines");
    me.theengines = props.globals.getNode("/systems/engines").getChildren("engine");
}

EngineControl.schedule = func {
    var selector = 0;

    me.speedkt = me.slave["asi"].getChild("indicated-speed-kt").getValue();
    me.speedmach = me.slave["mach"].getChild("indicated-mach").getValue();
    me.wow = me.slave["weight"].getChild("wow").getValue();

    for( var i = 0; i < constantaero.NBENGINES; i = i+1 ) {
         me.reheat = me.engcontrols[i].getChild("reheat").getValue();
         me.takeoff = me.rating.is_takeoff( i );
         me.climb = me.rating.is_climb( i );

         # auto
         if( me.thecontrol.getChild("schedule-auto").getValue() ) {
             selector = me.thecontrol.getChild("schedule").getValue();

             # detects bugs
             me.value = "";

             # normal
             if( selector == 0 ) {
                 if( me.setlow() ) {
                 }

                 elsif( !me.is_speedlow() and !me.reheat ) {
                     me.value = me.HIGH;
                 }

                 elsif( !me.is_speedlow() and me.climb ) {
                     me.value = me.MID;
                 }
             }

             # flyover
             elsif( selector < 0 ) {
                 if( me.setlow() ) {
                 }

                 elsif( !me.is_speedlow() and !me.is_speedhigh() and !me.reheat ) {
                     me.value = me.FLYOVER;
                 }

                 elsif( me.is_speedhigh() and !me.reheat ) {
                     me.value = me.HIGH;
                 }
             }

             # approach
             else {
                 if( me.wow ) {
                     me.value = me.LOW;
                 }

                 else {
                     me.value = me.MID;
                 }
             }
         }

         else {
             # force low
             if( me.thecontrol.getChild("schedule-low").getValue() ) {
                 me.value = me.LOW;
             }

             else {
                 if( me.slave["gear"].getChild("position-norm").getValue() == constantaero.GEARDOWN ) {
                     me.value = me.LOW
                 }

                 else {
                     if( me.takeoff and me.reheat ) {
                         me.value = me.LOW;
                     }

                     elsif( !me.takeoff and me.reheat ) {
                         me.value = me.MID;
                     }

                     # force high
                     else {
                         me.value = me.HIGH;
                     }
                 }
             }
         }

         me.theengines[i].getChild("schedule").setValue( me.value );
    }
}

EngineControl.setlow = func {
    var result = constant.TRUE;

    if( me.wow ) {
        me.value = me.LOW;
    }

    elsif( me.is_speedlow() ) {
        me.value = me.LOW;
    }

    elsif( me.takeoff and me.reheat ) {
        me.value = me.LOW;
    }

    else {
        result = constant.FALSE;
    }
}

EngineControl.is_speedlow = func {
    var result = constant.FALSE;

    if( me.speedkt <= me.LOWKT ) {
        result = constant.TRUE;
    }

    return result;
}

EngineControl.is_speedhigh = func {
    var result = constant.FALSE;

    if( me.speedmach >= me.HIGHMACH ) {
        result = constant.TRUE;
    }

    return result;
}


# =========
# ENGINE N1
# =========

EngineN1 = {};

EngineN1.new = func {
   var obj = { parents : [EngineN1,System],

           engall : nil,
           engcontrols : nil,
           engines : nil,
           enginesystem : nil,
           theengines : nil,

           THROTTLE88N1 : 0.806,                          # doesn't depend of temperature
           THROTTLEREHEAT : 0.10,

           N1REHEAT : 81,
           N1EXHAUST : 60,

           reheat : [ constant.FALSE, constant.FALSE, constant.FALSE, constant.FALSE ],

           texpath : "Textures",
           exhaust : [ 0.0, 0.0, 0.0, 0.0 ],

           engine4limiter : constant.FALSE,

           ENGINE4KT : 60.0
         };

   obj.init();

   return obj;
};

EngineN1.init = func {
    me.init_ancestor("/systems/engines");

    me.engall = props.globals.getNode("/controls/engines");
    me.engcontrols = props.globals.getNode("/controls/engines").getChildren("engine");
    me.engines = props.globals.getNode("/engines").getChildren("engine");
    me.enginesystem = props.globals.getNode("/systems/engines");
    me.theengines = props.globals.getNode("/systems/engines").getChildren("engine");
}

EngineN1.get_throttle = func( position ) {
    var maxthrottle = constantaero.THROTTLEMAX;

    if( me.engine4limiter ) {
        maxthrottle = me.THROTTLE88N1;
    }

    if( position > maxthrottle ) {
        position = maxthrottle;
    }

    return position;
}

EngineN1.schedule = func {
    var speedkt = me.slave["asi"].getChild("indicated-speed-kt").getValue();

    me.groundidle( speedkt );
    me.engine4( speedkt );
    me.reheatcontrol();
    me.exhausttexture();
}

EngineN1.groundidle = func( speedkt ) {
    var idle = constant.FALSE;

    if( me.engall.getChild("ground-idle14").getValue() or
        me.engall.getChild("ground-idle23").getValue() ) {
        idle = constant.TRUE;

        # only below 60 kt
        if( speedkt > me.ENGINE4KT ) {
            me.engall.getChild("ground-idle14").setValue( constant.FALSE );
            me.engall.getChild("ground-idle23").setValue( constant.FALSE );
            idle = constant.FALSE;
        }
    }

   # JSBSim can idle only 4 engines at once
   me.enginesystem.getChild("ground-idle").setValue(idle);
}

# Engine 4 N1 takeoff limiter
EngineN1.engine4 = func( speedkt ) {
    var throttle = 0.0;

    # avoids engine 4 vibration because of turbulences
    if( speedkt != nil ) {

        # only below 60 kt
        if( speedkt < me.ENGINE4KT ) {
            me.engine4limiter = me.engcontrols[3].getChild("n1-to-limiter").getValue();
        }

        # normal control
        else {
             if( me.engine4limiter ) {
                 me.engine4limiter = constant.FALSE;

                # align throttle
                throttle = me.engcontrols[2].getChild("throttle").getValue();
                me.engcontrols[3].getChild("throttle").setValue(throttle);
            }
        }
    }
}

EngineN1.reheatcontrol = func {
   var augmentation = constant.FALSE;

   for( var i = 0; i < constantaero.NBENGINES; i = i+1 ) {
        if( me.engcontrols[i].getChild("reheat").getValue() and
            me.engcontrols[i].getChild("throttle").getValue() > me.THROTTLEREHEAT and
            me.engines[i].getChild("n1").getValue() > me.N1REHEAT ) {
            augmentation = constant.TRUE;
        }
        else {
            augmentation = constant.FALSE;
        }

        if( me.reheat[i] != augmentation ) {
            me.reheat[i] = augmentation;
            me.engcontrols[i].getChild("augmentation").setValue( me.reheat[i] );
        }
   }
}

EngineN1.exhausttexture = func {
   var factor = 0.0;
   var texture = "";

   for( var i = 0; i < constantaero.NBENGINES; i = i+1 ) {
        if( me.engines[i].getChild("n1").getValue() > me.N1EXHAUST ) {
            factor = 1.0;
            texture = "concorde.rgb";
        }
        else {
            factor = 0.0;
            texture = me.texpath ~ "/concorde-nozzle.rgb";
        }

        if( me.exhaust[i] != factor ) {
            me.exhaust[i] = factor;
            me.theengines[i].getChild("nozzle-factor").setValue(me.exhaust[i]);
            me.theengines[i].getChild("nozzle-texture").setValue(texture);
        }
   }
}


# ======
# RATING
# ======

Rating = {};

Rating.new = func {
   var obj = { parents : [Rating],

           enginen1 : EngineN1.new(),

           autothrottles : nil,
           engcontrols : nil,
           gears : nil,
           theengines : nil,

# contingency is not yet supported
           THROTTLETAKEOFF : 1.0,                         # N2 105.7 % (106.0 in Engines file)
           THROTTLECLIMB : 0.980,                         # N2 105.1 %
           THROTTLECRUISE : 0.967,                        # N2 104.5 % (guess)

           GEARLEFT : 1,
           GEARRIGHT : 3
         };

   obj.init();

   return obj;
};

Rating.init = func {
   me.autothrottles = props.globals.getNode("/autopilot/locks/autothrottle").getChildren("engine");
   me.engcontrols = props.globals.getNode("/controls/engines").getChildren("engine");
   me.gears = props.globals.getNode("/gear").getChildren("gear");
   me.theengines = props.globals.getNode("/systems/engines").getChildren("engine");
}

Rating.set_throttle = func( position ) {
   var rating = "";
   var maxthrottle = 0.0;
   var position_new = 0.0;

   # faster to process here
   for( var i = 0; i < constantaero.NBENGINES; i = i+1 ) {
        position_new = position;
        rating = me.engcontrols[i].getChild("rating").getValue();

        # autoland first
        if( rating == "takeoff" ) {
            maxthrottle = me.THROTTLETAKEOFF;
        }

        # flight
        else {
            rating = me.engcontrols[i].getChild("rating-flight").getValue();

            if( rating == "climb" ) {
                maxthrottle = me.THROTTLECLIMB;
            }

            # cruise
            else {
                maxthrottle = me.THROTTLECRUISE;
            }
        }

        if( position_new > maxthrottle ) {
            position_new = maxthrottle;
        }

        # engine N1 limiter
        if( i == 3 ) {
            position_new = me.enginen1.get_throttle( position_new );
        }

        # default, except autothrottle
        if( me.autothrottles[i].getValue() == "" ) {
            me.engcontrols[i].getChild("throttle").setValue( position_new );
        }

         # last human operation (to detect goaround).
        me.engcontrols[i].getChild("throttle-manual").setValue( position );
   }
}

Rating.schedule = func {
   var ratingnow = "";
   var j = 0;
   var rating = constant.FALSE;
   var reheat = constant.FALSE;
   var monitor = getprop("/instrumentation/takeoff-monitor/armed");

   me.enginen1.schedule();

   # arm takeoff rating
   for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
        ratingnow = me.engcontrols[i].getChild("rating").getValue();
        if( ratingnow != "takeoff" ) {
            # engines 2 and 3 by right gear
            if( i > 0 and i < 3 ) {
                j = me.GEARRIGHT;
            }

            # engines 1 and 4 armed by left gear
            else {
                j = me.GEARLEFT;
            }

            if( me.gears[j].getChild("position-norm").getValue() == constantaero.GEARDOWN ) {
                me.engcontrols[i].getChild("rating").setValue("takeoff");
            }
        }
   }

   for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
       rating = me.engcontrols[i].getChild("rating").getValue();
       reheat = me.engcontrols[i].getChild("reheat").getValue();

       if( reheat and rating == "takeoff" ) {
           # automatic contigency, if takeoff monitor
           if( monitor ) {
               me.engcontrols[i].getChild("contingency").setValue(constant.TRUE);
           }
       }
       elsif( !reheat and me.engcontrols[i].getChild("contingency").getValue() ) {
           me.engcontrols[i].getChild("contingency").setValue(constant.FALSE);
       }
   }

   # apply to engines
   for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
        rating = me.engcontrols[i].getChild("rating").getValue();
        if( rating != "takeoff" ) {
            rating = me.engcontrols[i].getChild("rating-flight").getValue();
        }
        me.theengines[i].getChild("rating").setValue(rating);
   }
}

Rating.is_takeoff = func( index ) {
    var result = constant.FALSE;

    if( me.engcontrols[index].getChild("rating").getValue() == "takeoff" ) {
        result = constant.TRUE;
    }

    return result;
}

Rating.is_climb = func( index ) {
    var result = constant.FALSE;

    if( !me.is_takeoff( index ) and
        me.engcontrols[index].getChild("rating-flight").getValue() == "climb" ) {
        result = constant.TRUE;
    }

    return result;
}


# =======================
# SECONDARY NOZZLE BUCKET
# =======================

Bucket = {};

Bucket.new = func {
   var obj = { parents : [Bucket,System],

           TRANSITSEC : 6.0,                                   # reverser transit in 6 s
           BUCKETSEC : 1.0,                                    # refresh rate
           RATEDEG : 0.0,                                      # maximum rotation speed
           REVERSERDEG : 73.0,
           TAKEOFFDEG : 21.0,
           SUPERSONICDEG : 0.0,
           SUBSONICMACH : 0.55,
           SUPERSONICMACH : 1.1,
           COEF : 0.0,

           propulsions : nil,
           engcontrols : nil,
           engines : nil
         };

   obj.init();

   return obj;
};

Bucket.set_rate = func( rates ) {
   var offsetdeg = me.REVERSERDEG - me.TAKEOFFDEG;

   me.BUCKETSEC = rates;

   me.RATEDEG = offsetdeg * ( me.BUCKETSEC / me.TRANSITSEC );
}

Bucket.init = func {
   me.init_ancestor("/systems/engines");

   var denom = me.SUPERSONICMACH - me.SUBSONICMACH;

   me.propulsions = props.globals.getNode("/fdm/jsbsim/propulsion").getChildren("engine");

   me.engines = props.globals.getNode("/systems/engines").getChildren("engine");
   me.engcontrols = props.globals.getNode("/controls/engines").getChildren("engine");

   me.set_rate( me.BUCKETSEC );

   me.COEF = me.TAKEOFFDEG / denom;
}

Bucket.schedule = func {
    me.position();
}

Bucket.increase = func( angledeg, maxdeg ) {
    angledeg = angledeg + me.RATEDEG;
    if( angledeg > maxdeg ) {
        angledeg = maxdeg;
    }

    return angledeg;
}

Bucket.decrease = func( angledeg, mindeg ) {
    angledeg = angledeg - me.RATEDEG;
    if( angledeg < mindeg ) {
        angledeg = mindeg;
    }

    return angledeg;
}

Bucket.apply = func( property, angledeg, targetdeg ) {
   var offsetdeg = 0.0;
   var valuedeg = 0.0;

   if( angledeg != targetdeg ) {
       offsetdeg = targetdeg - angledeg;
       if( offsetdeg > 0 ) {
           valuedeg = me.increase( angledeg, targetdeg );
       }
       else {
           valuedeg = me.decrease( angledeg, targetdeg );
       }
       interpolate( property, valuedeg, me.BUCKETSEC );
   }
}

# bucket position
Bucket.position = func {
   var step = 0.0;
   var angledeg = 0.0;
   var valuedeg = 0.0;
   var result = 0.0;
   var speedmach = me.slave["mach"].getChild("indicated-mach").getValue();

   # supersonic : 0 deg
   var bucketdeg = me.SUPERSONICDEG;

   # takeoff : 21 deg
   if( speedmach < me.SUBSONICMACH ) {
       bucketdeg = me.TAKEOFFDEG;
   }
   # subsonic : 21 to 0 deg
   elsif( speedmach <= me.SUPERSONICMACH ) {
       step = speedmach - me.SUBSONICMACH;
       bucketdeg = me.TAKEOFFDEG - me.COEF * step;
   }

   for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
       # CAUTION : use controls, because there is a delay by /engines/engine[0]/reversed !
       if( me.engcontrols[i].getChild("reverser").getValue() ) {
           # reversed : 73 deg
           angledeg = me.REVERSERDEG;
           valuedeg = angledeg;
       }
       else {
           angledeg = 0.0;
           valuedeg = bucketdeg;
       }

       result = me.engines[i].getChild("bucket-deg").getValue();
       me.apply( "/systems/engines/engine[" ~ i ~ "]/bucket-deg", result, valuedeg );
# reverser was implemented by 0.9.9
#       result = me.propulsions[i].getChild("reverser-angle").getValue();
#       me.apply( "/fdm/jsbsim/propulsion/engine[" ~ i ~ "]/reverser-angle", result, angledeg );
   }
}


# ===================
# SECONDARY AIR DOORS
# ===================

AirDoor = {};

AirDoor.new = func {
   var obj = { parents : [AirDoor,System],

           ENGINESMACH : 0.26,
           ENGINE4KT : 220.0,

           engines : nil,
           engcontrols : nil
         };

   obj.init();

   return obj;
};

AirDoor.init = func {
   me.engines = props.globals.getNode("/systems/engines").getChildren("engine");
   me.engcontrols = props.globals.getNode("/controls/engines").getChildren("engine");

   me.init_ancestor("/systems/engines");
}

# air door position
AirDoor.schedule = func {
   var value = constant.FALSE;
   var speedmach = me.slave["mach"].getChild("indicated-mach").getValue();
   var speedkt = me.slave["asi"].getChild("indicated-speed-kt").getValue();
   var touchdown = getprop("/instrumentation/weight-switch/wow");
   var gearpos = getprop("/gear/gear[1]/position-norm");

   # engines 1 to 3 :
   for( var i=0; i<3; i=i+1 ) {
       if( me.engcontrols[i].getChild("secondary-air-door").getValue() ) {
           value = me.engines[i].getChild("secondary-air-door").getValue();
           # opens above Mach 0.26
           if( !value ) {
               if( speedmach > me.ENGINESMACH ) {
                   value = constant.TRUE;
               }
           }
           # shuts below Mach 0.26, if touch down
           elsif( speedmach < me.ENGINESMACH and touchdown ) {
               value = constant.FALSE;
           }
           me.engines[i].getChild("secondary-air-door").setValue(value);
       }
   }

   # engine 4
   if( me.engcontrols[3].getChild("secondary-air-door").getValue() ) {
       value = me.engines[3].getChild("secondary-air-door").getValue();
       # opens above 220 kt
       if( !value ) {
           if( speedkt > me.ENGINE4KT ) {
               value = constant.TRUE;
           }
       } 
       # shuts below Mach 0.26, gear down
       elsif( speedmach < me.ENGINESMACH and gearpos == constantaero.GEARDOWN ) {
           value = constant.FALSE;
       }
       me.engines[3].getChild("secondary-air-door").setValue(value);
   }
}


# ===========
# INTAKE RAMP
# ===========

Intake = {};

Intake.new = func {
   var obj = { parents : [Intake,System],

           MAXRAMP : 50.0,
           MINRAMP : 0.0,
           MAXMACH : 2.02,
           MINMACH : 1.3,
           INLETMACH : 0.75,
           OFFSETMACH : 0.0,

           LANEA : 2,
           LANEAUTOA : 0,

           POSSUBSONIC : 1.0,
           POSSUPERSONIC : 0.0,

           enginecontrol : nil,
           enginesystem : nil,
           engines : nil,

           hydmain : [ "green", "green", "blue", "blue" ],

           lane : [ constant.TRUE, constant.FALSE ]
         };

   obj.init();

   return obj;
};

Intake.init = func {
   me.init_ancestor("/systems/engines");

   me.enginecontrol = props.globals.getNode("/controls/engines").getChildren("engine");
   me.enginesystem = props.globals.getNode("/systems/engines");
   me.engines = props.globals.getNode("/systems/engines").getChildren("engine");

   me.OFFSETMACH = me.MAXMACH - me.MINMACH;
}

# main system failure
Intake.amber_intake = func( index ) {
    var result = constant.FALSE;

    # auto or green / blue selected
    if( !me.slave["hydraulic"].getChild(me.hydmain[index]).getValue() and
        me.engines[index].getChild("intake-main").getValue() ) {
        result = constant.TRUE;
    }

    # yellow selected
    elsif( !me.slave["hydraulic"].getChild("yellow").getValue() and
           me.engines[index].getChild("intake-standby").getValue() ) {
        result = constant.TRUE;
    }

    return result;
}

Intake.laneexport = func {
   var selector = 0;

   for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
        selector = me.enginecontrol[i].getChild("intake-selector").getValue();

        if( selector == me.LANEAUTOA or selector == me.LANEA ) {
            me.lane[0] = constant.TRUE;
            me.lane[1] = constant.FALSE;
        }
        else {
            me.lane[0] = constant.FALSE;
            me.lane[1] = constant.TRUE;
        }

        for( var j=0; j<constantaero.NBAUTOPILOTS; j=j+1 ) {
             me.engines[i].getChild("intake-lane", j).setValue(me.lane[j]);
        }
   }
}

Intake.schedule = func {
   var speedmach = me.slave["mach"].getChild("indicated-mach").getValue();

   me.auxilliaryinlet( speedmach );
   me.ramphydraulic();
   me.rampposition( speedmach );
}

Intake.auxilliaryinlet = func( speedmach ) {
   var pos = constant.FALSE;

   if( speedmach < me.INLETMACH ) {
       pos = constant.TRUE;
   }

   for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
        me.engines[i].getChild("intake-aux-inlet").setValue(pos);
   }
}

Intake.ramphydraulic = func {
   var main = constant.FALSE;
   var standby  = constant.FALSE;

   for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
        if( me.enginecontrol[i].getChild("intake-auto").getValue() ) {
            main = constant.TRUE;
            standby = !me.slave["hydraulic"].getChild(me.hydmain[i]).getValue();
        }
        else {
            main = me.enginecontrol[i].getChild("intake-main").getValue();
            standby = !main;
        }

        me.engines[i].getChild("intake-main").setValue(main);
        me.engines[i].getChild("intake-standby").setValue(standby);
   }
}

Intake.rampposition = func( speedmach ) {
   var stepmach = 0.0;
   var coef = 0.0;
   var pospercent = 0.0;
   var ratio = 0.0;
   var ratiopercent = 0.0;
   var ramppercent = me.MAXRAMP;
   var rampsubsonic = me.POSSUPERSONIC;
   var hydfailure = constant.FALSE;

   if( speedmach <= me.MINMACH ) {
       ramppercent = me.MINRAMP;
       rampsubsonic = me.POSSUBSONIC;
   }
   elsif( speedmach > me.MINMACH and speedmach < me.MAXMACH ) {
       stepmach = speedmach - me.MINMACH;
       coef = stepmach / me.OFFSETMACH;
       ramppercent = me.MAXRAMP * coef;
       rampsubsonic = me.POSSUPERSONIC;
   }

   for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
        if( me.amber_intake(i) ) {
            hydfailure = constant.TRUE;
            break;
        }
   }

   # TO DO : effect of throttle on intake pressure ratio error

   # engineer moves ramp manually
   if( hydfailure ) {
       for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
            pospercent = me.engines[i].getChild("ramp-percent").getValue();

            # to the left (negativ), if throttle lever must be retarded
            ratio = ( ramppercent - pospercent ) / me.MAXRAMP;

            ratiopercent = ratio * 100;
            me.engines[i].getChild("intake-ratio-error").setValue(ratiopercent);
       }

       # ramp is too much closed (supercritical)
       if( ratio < 0.0 ) {
           if( rampsubsonic == me.POSSUBSONIC ) {
               rampsubsonic = me.superramp( ratio, me.POSSUPERSONIC, rampsubsonic );
           }
           else {
               rampsubsonic = me.superramp( ratio, me.POSSUBSONIC, rampsubsonic );
           }
       }

       # ramp is too much opened (subcritical)
       elsif( ratio > 0.0 ) {
           if( rampsubsonic == me.POSSUPERSONIC ) {
               rampsubsonic = me.subramp( ratio, me.POSSUBSONIC, rampsubsonic );
           }
           else {
               rampsubsonic = me.subramp( ratio, me.POSSUPERSONIC, rampsubsonic );
           }
       }
   }

   # hydraulics moves intake ramp
   else {
       for( var i=0; i<constantaero.NBENGINES; i=i+1 ) {
            me.engines[i].getChild("ramp-percent").setValue(ramppercent);
            me.engines[i].getChild("intake-ratio-error").setValue(0.0);
       }
   }

   # JSBSim can disable only 4 intakes at once
   me.enginesystem.getChild("intake-subsonic").setValue(rampsubsonic);
}

Intake.superramp = func( ratio, target, present ) {
   var result = present - ( target - present ) * ratio;

   return result;
}

Intake.subramp = func( ratio, target, present ) {
   var result = present + ( target - present ) * ratio;

   return result;
}
