/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Collections;

using PostgreSql.Data.NPgClient;

namespace PostgreSql.Data.PgSqlClient
{
	public sealed class PgTransaction : MarshalByRefObject, IDbTransaction, IDisposable
	{
		#region Fields

		private PgConnection		connection;
		private IsolationLevel		isolationLevel;
		private bool				disposed;
		private bool				isUpdated;

		#endregion

		#region Properties

		IDbConnection IDbTransaction.Connection
		{
			get { return Connection; }
		}

		public PgConnection Connection
		{
			get 
			{ 
				if (!this.isUpdated)
				{
					return this.connection; 
				}
				else
				{
					return null;
				}
			}
		}

		public IsolationLevel IsolationLevel 
		{
			get { return isolationLevel; }
		}

		internal bool IsUpdated
		{
			get { return isUpdated; }
			set 
			{ 
				if (connection != null &&
					value)
				{
					connection.ActiveTransaction	= null;
					connection						= null;
				}
				isUpdated = value; 
			}
		}

		#endregion

		#region Constructors

		private PgTransaction()
		{
			isolationLevel	= IsolationLevel.ReadCommitted;
		}
		
		internal PgTransaction(PgConnection connection) : this(connection, IsolationLevel.ReadCommitted)
		{
		}

		internal PgTransaction(PgConnection connection, IsolationLevel isolation)
		{
			this.connection		= connection;
			this.isolationLevel = isolation;
		}				

		#endregion

		#region Finalizer

		~PgTransaction()
		{
			this.Dispose(false);
		}

		#endregion

		#region IDisposable Methods

		public void Dispose()
		{
			this.Dispose(true);
			System.GC.SuppressFinalize(this);
		}

		private void Dispose(bool disposing)
		{
			if (!disposed)			
			{
				if (disposing)
				{
					try
					{
						if (connection != null && !isUpdated)
						{
							// Implicitly roll back if the transaction still valid.
							this.Rollback();
						}
					}
					finally
					{
						if (connection != null)
						{
							connection.ActiveTransaction	= null;
							connection						= null;
						}
						disposed	= true;
						isUpdated	= true;
					}
				}
			}			
		}
		
		#endregion

		#region Methods

		public void Commit()
		{
			if (isUpdated)
			{
				throw new InvalidOperationException("This Transaction has completed; it is no longer usable.");
			}
			if (Connection.DataReader != null)
			{
				throw new InvalidOperationException("PgCommand is currently busy Open, Fetching.");
			}

			try
			{
				connection.DbConnection.DB.CommitTransaction();
				
				IsUpdated = true;
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}
		}

		public void Rollback()
		{
			if (isUpdated)
			{
				throw new InvalidOperationException("This Transaction has completed; it is no longer usable.");
			}
			if (Connection.DataReader != null)
			{
				throw new InvalidOperationException("PgCommand is currently busy Open, Fetching.");
			}

			try
			{
				connection.DbConnection.DB.RollbackTransction();
				
				IsUpdated = true;
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}
		}

		internal void InternalBeginTransaction()
		{
			try
			{
				connection.DbConnection.DB.BeginTransaction(isolationLevel);

				IsUpdated = false;
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}
		}

		#endregion
	}
}
