/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Text;
using System.Data;
using System.Collections;

namespace PostgreSql.Data.NPgClient
{
	#region Enumerations

	internal enum PgStatementStatus
	{
		Initial,
		Parsed,
		Described,
		Binded,
		Executed,
		OnQuery,
		Error
	}

	#endregion

	internal class PgStatement
	{
		#region Fields

		private PgDbClient		db;

		private string			stmtText;
		private bool			hasRows;
		private string			tag;
		private string			parseName;
		private string			portalName;
		private int				fetchSize;
		private bool			allRowsFetched;
		private PgRowDescriptor rowDescriptor;
		private object[]		rows;
		private int				rowIndex;
		private PgParameter[]	parameters;
		private PgParameter		outParameter;
		private int				recordsAffected;
		private char			transactionStatus;
		private PgStatementStatus status;

		#endregion

		#region Properties

		public PgDbClient DbHandle
		{
			get { return db; }
			set { db = value; }
		}

		public string StmtText
		{
			get { return stmtText; }
			set { stmtText = value; }
		}

		public bool HasRows
		{
			get { return hasRows; }
		}

		public string Tag		
		{
			get { return tag; }
		}

		public string ParseName
		{
			get { return parseName; }
			set { parseName = value; }
		}

		public string PortalName
		{
			get { return portalName; }
			set { portalName = value; }
		}

		public PgRowDescriptor RowDescriptor
		{
			get { return rowDescriptor; }
		}

		public object[] Rows
		{
			get { return rows; }
		}

		public PgParameter[] Parameters
		{
			get { return parameters; }
		}

		public PgParameter OutParameter
		{
			get { return outParameter; }
			set { outParameter = value; }
		}

		public int RecordsAffected
		{
			get { return recordsAffected; }
		}

		public PgStatementStatus Status
		{
			get { return status; }
		}

		public char TransactionStatus
		{
			get { return transactionStatus; }
		}

		#endregion

		#region Constructors

		public PgStatement()
		{
			this.outParameter		= new PgParameter();
			this.rows				= null;
			this.rowIndex			= 0;
			this.parseName			= String.Empty;
			this.portalName			= String.Empty;
			this.recordsAffected	= -1;
			this.status				= PgStatementStatus.Initial;
			this.fetchSize			= 200;
		}

		public PgStatement(PgDbClient db) : this()
		{
			this.db = db;
		}

		public PgStatement(PgDbClient db, string parseName, string portalName) : this(parseName, portalName)
		{
			this.db = db;
		}

		public PgStatement(PgDbClient db, string parseName, string portalName, string stmtText) : this(db, parseName, portalName)
		{
			this.stmtText = stmtText;
		}

		public PgStatement(PgDbClient db, string stmtText) : this(db)
		{
			this.stmtText = stmtText;
		}

		public PgStatement(string parseName, string portalName) : this()
		{
			this.parseName	= parseName;
			this.portalName = portalName;
		}

		#endregion

		#region Methods

		public void Parse()
		{
			lock (db)
			{
				try
				{
					// Clear actual row list
					rows		= null;
					rowIndex	= 0;

					// Initialize RowDescriptor and Parameters
					rowDescriptor	= new PgRowDescriptor(0);
					parameters		= new PgParameter[0];

					PgOutputPacket packet = new PgOutputPacket(db.Settings.Encoding);

					packet.WriteString(ParseName);
					packet.WriteString(stmtText);
					packet.WriteShort(0);

					// Send packet to the server
					db.SendPacket(packet, PgFrontEndCodes.PARSE);
					
					// Update status
					this.status	= PgStatementStatus.Parsed;
				}
				catch (PgClientException ex)
				{
					// Update status
					this.status	= PgStatementStatus.Error;
					// Throw exception
					throw ex;
				}
			}
		}

		public void Describe()
		{
			describe('S');
		}

		public void DescribePortal()
		{
			describe('P');
		}

		private void describe(char stmtType)
		{
			lock (db)
			{
				try
				{					
					string name = ((stmtType == 'S') ? ParseName : PortalName);

					PgOutputPacket packet = new PgOutputPacket(db.Settings.Encoding);

					packet.Write((byte)stmtType);
					packet.WriteString(name);

					// Send packet to the server
					db.SendPacket(packet, PgFrontEndCodes.DESCRIBE);

					// Flush pending messages
					db.Flush();

					// Receive Describe response
					PgResponsePacket response = new PgResponsePacket();
					while ((response.Message != PgBackendCodes.ROW_DESCRIPTION &&
						response.Message != PgBackendCodes.NODATA))
					{
						response = db.ReceiveResponsePacket();
						processSqlPacket(response);
					}

					// Update status
					this.status	= PgStatementStatus.Described;
				}
				catch (Exception ex)
				{
					// Update status
					this.status	= PgStatementStatus.Error;
					// Throw exception
					throw ex;
				}
			}
		}

		public void Bind()
		{
			lock (db)
			{
				try
				{
					PgOutputPacket packet = new PgOutputPacket(db.Settings.Encoding);

					// Destination portal name
					packet.WriteString(PortalName);
					// Prepared statement name
					packet.WriteString(ParseName);
					
					// Send parameters format code.
					packet.WriteShort((short)parameters.Length);
					for (int i = 0; i < parameters.Length; i++)
					{
						packet.WriteShort(parameters[i].DataType.FormatCode);
					}

					// Send parameter values
					packet.WriteShort((short)parameters.Length);
					for (int i = 0; i < parameters.Length; i++)
					{
						packet.WriteParameter(parameters[i]);
					}
					
					// Send column information
					packet.WriteShort((short)rowDescriptor.Fields.Length);
					for (int i = 0; i < rowDescriptor.Fields.Length; i++)
					{
						packet.WriteShort(rowDescriptor.Fields[i].DataType.FormatCode);
					}

					// Send packet to the server
					db.SendPacket(packet, PgFrontEndCodes.BIND);

					// Update status
					this.status	= PgStatementStatus.Parsed;
				}
				catch (Exception ex)
				{
					// Update status
					this.status	= PgStatementStatus.Error;
					// Throw exception
					throw ex;
				}
			}
		}

		public void Execute()
		{
			lock (db)
			{
				try
				{
					PgOutputPacket packet = new PgOutputPacket(db.Settings.Encoding);

					packet.WriteString(PortalName);
					packet.WriteInt(fetchSize);	// Rows to retrieve ( 0 = nolimit )

					// Send packet to the server
					db.SendPacket(packet, PgFrontEndCodes.EXECUTE);

					// Flush pending messages
					db.Flush();
					
					// Receive response
					PgResponsePacket response = new PgResponsePacket();
					while (response.Message != PgBackendCodes.READY_FOR_QUERY &&
						response.Message != PgBackendCodes.PORTAL_SUSPENDED &&
						response.Message != PgBackendCodes.COMMAND_COMPLETE)
					{
						response = db.ReceiveResponsePacket();
						processSqlPacket(response);
					}

					// reset rowIndex
					this.rowIndex = 0;

					// If the command is finished and has returned rows
					// set all rows are received
					if ((response.Message == PgBackendCodes.READY_FOR_QUERY ||
						response.Message == PgBackendCodes.COMMAND_COMPLETE) &&
						this.hasRows)
					{
						this.allRowsFetched = true;
					}

					// If all rows are received or the command doesn't return
					// rows perform a Sync.
					if (!this.hasRows || this.allRowsFetched)
					{
						db.Sync();
					}

					// Update status
					this.status	= PgStatementStatus.Executed;
				}
				catch (Exception ex)
				{
					// Update status
					this.status	= PgStatementStatus.Error;
					// Throw exception
					throw ex;
				}
			}
		}

		public void ExecuteFunction(int id)
		{
			lock (db)
			{
				try
				{
					PgOutputPacket packet = new PgOutputPacket(db.Settings.Encoding);

					// Function id
					packet.WriteInt(id);

					// Send parameters format code.
					packet.WriteShort((short)parameters.Length);
					for (int i = 0; i < parameters.Length; i++)
					{
						packet.WriteShort(parameters[i].DataType.FormatCode);
					}

					// Send parameter values
					packet.WriteShort((short)parameters.Length);
					for (int i = 0; i < parameters.Length; i++)
					{
						packet.WriteParameter(parameters[i]);
					}

					// Send the format code for the function result
					packet.WriteShort(PgCodes.BINARY_FORMAT);

					// Send packet to the server
					db.SendPacket(packet, PgFrontEndCodes.FUNCTION_CALL);
					
					// Receive response
					PgResponsePacket response = new PgResponsePacket();
					while (response.Message != PgBackendCodes.READY_FOR_QUERY)
					{
						response = db.ReceiveResponsePacket();
						processSqlPacket(response);
					}

					// Update status
					this.status	= PgStatementStatus.Executed;
				}
				catch (Exception ex)
				{
					// Update status
					this.status	= PgStatementStatus.Error;
					// Throw exception
					throw ex;
				}
			}
		}

		public void Query()
		{
			ArrayList innerRows = new ArrayList();

			lock (db)
			{
				try
				{
					PgOutputPacket packet = new PgOutputPacket(db.Settings.Encoding);

					packet.WriteString(stmtText);

					// Send packet to the server
					db.SendPacket(packet, PgFrontEndCodes.QUERY);

					// Update Status
					this.status = PgStatementStatus.OnQuery;

					// Set fetch size
					this.fetchSize = 1;

					// Receive response
					PgResponsePacket response = new PgResponsePacket();
					while (response.Message != PgBackendCodes.READY_FOR_QUERY)
					{
						response = db.ReceiveResponsePacket();
						processSqlPacket(response);

						if (this.hasRows && 
							response.Message == PgBackendCodes.DATAROW)
						{
							innerRows.Add(this.rows[0]);
							this.rowIndex = 0;
						}
					}
					
					if (this.hasRows)
					{
						// Obtain all the rows
						this.rows = (object[])innerRows.ToArray(typeof(object));

						// reset rowIndex
						this.rowIndex = 0;

						// Set allRowsFetched flag
						this.allRowsFetched = true;
					}

					// reset fetch size
					this.fetchSize = 200;

					// Update status
					this.status	= PgStatementStatus.Executed;
				}
				catch (Exception ex)
				{
					// Update status
					this.status	= PgStatementStatus.Error;
					// Throw exception
					throw ex;
				}
			}
		}

		public object[] FetchRow()
		{
			object[] row = null;

			if ((!this.allRowsFetched && this.rows == null) ||
				(!this.allRowsFetched && this.rows.Length == 0) ||
				(!this.allRowsFetched && this.rowIndex >= this.fetchSize))
			{
				lock (this)
				{
					// Retrieve next group of rows
					this.Execute();
				}
			}
			
			if (this.rows != null	&& 
				this.rows.Length > 0 &&
				this.rows[this.rowIndex] != null)
			{
				// Return always first row
				row = (object[])this.rows[this.rowIndex++];
			}

			if (this.rows != null &&
				(this.rowIndex >= this.fetchSize ||
				this.rowIndex >= this.rows.Length ||
				this.rows[this.rowIndex] == null))
			{
				this.rows = null;
			}

			return row;
		}

		public void Close()
		{
			close('S');
		}

		public void ClosePortal()
		{
			close('P');
		}

		private void close(char stmtType)
		{
			lock (db)
			{
				try
				{
					string name = ((stmtType == 'S') ? ParseName : PortalName);

					PgOutputPacket packet = new PgOutputPacket(db.Settings.Encoding);

					packet.Write((byte)stmtType);
					packet.WriteString(name);

					// Send packet to the server
					db.SendPacket(packet, PgFrontEndCodes.CLOSE);

					// Sync server and client
					db.Flush();

					// Read until CLOSE COMPLETE message is received
					PgResponsePacket response = new PgResponsePacket();
					while (response.Message != PgBackendCodes.CLOSE_COMPLETE)
					{
						response = db.ReceiveResponsePacket();
						processSqlPacket(response);
					}

					// Clear rows
					rows		= null;
					rowIndex	= 0;

					// Update Status
					status = PgStatementStatus.Initial;
				}
				catch (Exception ex)
				{
					// Update Status
					status = PgStatementStatus.Error;
					// Throw exception
					throw ex;
				}
			}
		}

		#endregion

		#region Misc Methods

		public string GetPlan(bool verbose)
		{
			lock (db)
			{
				try
				{
					PgStatement getPlan = new PgStatement();

					getPlan.DbHandle	= this.db;
					getPlan.StmtText	= "EXPLAIN ANALYZE ";
					if (verbose)
					{
						getPlan.StmtText += "VERBOSE ";
					}
					getPlan.StmtText += stmtText;

					getPlan.Query();

					StringBuilder stmtPlan = new StringBuilder();

					foreach (object[] row in getPlan.Rows)
					{
						stmtPlan.AppendFormat("{0} \r\n", row[0]);
					}

					getPlan.Close();

					return stmtPlan.ToString();
				}
				catch (PgClientException ex)
				{
					throw ex;
				}
			}
		}

		#endregion

		#region Response Methods

		private void processSqlPacket(PgResponsePacket packet)
		{
			switch (packet.Message)
			{
				case PgBackendCodes.READY_FOR_QUERY:
					transactionStatus = packet.ReadChar();
					break;

				case PgBackendCodes.FUNCTION_CALL_RESPONSE:
					processFunctionResult(packet);
					break;

				case PgBackendCodes.ROW_DESCRIPTION:
					processRowDescription(packet);
					break;

				case PgBackendCodes.DATAROW:
					this.hasRows = true;
					processDataRow(packet);
					break;

				case PgBackendCodes.EMPTY_QUERY_RESPONSE:
				case PgBackendCodes.NODATA:
					this.hasRows	= false;
					this.rows		= null;
					this.rowIndex	= 0;
					break;

				case PgBackendCodes.COMMAND_COMPLETE:
					processTag(packet);
					break;

				case PgBackendCodes.PARAMETER_DESCRIPTION:
					processParameterDescription(packet);
					break;

				case PgBackendCodes.BIND_COMPLETE:
				case PgBackendCodes.PARSE_COMPLETE:
				case PgBackendCodes.CLOSE_COMPLETE:
					break;
			}
		}

		private void processTag(PgResponsePacket packet)
		{
			string[] elements = null;

			tag = packet.ReadNullString();
			
			elements = tag.Split(' ');

			switch (elements[0])
			{
				case "FETCH":
				case "SELECT":
					recordsAffected = -1;
					break;

				case "INSERT":
					recordsAffected = Int32.Parse(elements[2]);
					break;

				case "UPDATE":
				case "DELETE":
				case "MOVE":
					recordsAffected = Int32.Parse(elements[1]);
					break;
			}
		}

		private void processFunctionResult(PgResponsePacket packet)
		{
			int length = packet.ReadInt();

			outParameter.Value = packet.ReadValue(outParameter.DataType, length);
		}

		private void processRowDescription(PgResponsePacket packet)
		{
			rowDescriptor = new PgRowDescriptor(packet.ReadShort());

			for (int i = 0; i < rowDescriptor.Fields.Length; i++)
			{
				rowDescriptor.Fields[i] = new PgFieldDescriptor();

				rowDescriptor.Fields[i].FieldName		= packet.ReadNullString();
				rowDescriptor.Fields[i].OidTable		= packet.ReadInt();
				rowDescriptor.Fields[i].OidNumber		= packet.ReadShort();
				rowDescriptor.Fields[i].DataType		= PgDbClient.Types[packet.ReadInt()];
				rowDescriptor.Fields[i].DataTypeSize	= packet.ReadShort();
				rowDescriptor.Fields[i].TypeModifier	= packet.ReadInt();
				rowDescriptor.Fields[i].FormatCode		= packet.ReadShort();
			}
		}

		private void processParameterDescription(PgResponsePacket packet)
		{
			parameters = new PgParameter[packet.ReadShort()];

			for (int i = 0; i < parameters.Length; i++)
			{
				parameters[i] = new PgParameter(packet.ReadInt());
			}
		}
		
		private void processDataRow(PgResponsePacket packet)
		{
			int			fieldCount	= packet.ReadShort();
			object[]	values		= new object[fieldCount];

			if (this.rows == null)
			{
				this.rows		= new object[fetchSize];
				this.rowIndex	= 0;
			}

			for (int i = 0; i < values.Length; i++)
			{
				int length = packet.ReadInt();

				switch (length)
				{
					case -1:
						values[i] = System.DBNull.Value;
						break;

					default:
						if (this.status == PgStatementStatus.OnQuery)
						{
							values[i] = packet.ReadValueFromString(
								rowDescriptor.Fields[i].DataType,
								length);
						}
						else
						{
							values[i] = packet.ReadValue(
								rowDescriptor.Fields[i].DataType,
								length);
						}
						break;
				}
			}

			this.rows[this.rowIndex++] = values;
		}

		#endregion
	}
}