// (c) Microsoft Corporation 2005-2007.

#light

namespace Microsoft.FSharp.Core

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Compatibility

[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module String = 

    let concat sep (strings : string list) =  System.String.Join(sep, CompatArray.of_list strings)
    let get (str:string) i = 
      try str.[i]
      with :? System.ArgumentException -> invalid_arg "String.get" 
    let length (str:string) = str.Length
    let sub (s:string) (start:int) (len:int) =  
      try s.Substring(start,len)
      with :? System.ArgumentException -> failwith "String.sub" 
    let iter (f : (char -> unit)) (s:string) =
      for i = 0 to s.Length - 1 do 
        f s.[i] 
      done

    let map (f: char -> char) (s:string) = 
        let res = new System.Text.StringBuilder(s.Length) in
        s |> iter (fun c -> res.Append(f c) |> ignore);
        res.ToString()

    let map_concat (f: char -> string) (s:string) = 
        let res = new System.Text.StringBuilder(s.Length) in
        s |> iter (fun c -> res.Append(f c) |> ignore);
        res.ToString()

    let for_all f (s:string) =
      let rec check i = (i >= s.Length) || (f (get s i) && check (i+1)) in 
      check 0

    let exists f (s:string) =
      let rec check i = (i < s.Length) && (f (get s i) || check (i+1)) in 
      check 0  

    let compare (x:string) y = compare x y

    let fast_get (s:string) n = s.[n]
    let of_char (c:char) = System.Char.ToString(c)
    let make (n: int) (c: char) : string = new System.String(c, n)

    let index_from (s:string) (start:int) (c:char) =  
      try let r = s.IndexOf(c,start) in if r = -1 then not_found() else r
      with :? System.ArgumentException -> invalid_arg "String.index_from" 
      
    let rindex_from (s:string) (start:int) (c:char) =  
      try let r =  s.LastIndexOf(c,start) in if r = -1 then not_found() else r
      with :? System.ArgumentException -> invalid_arg "String.rindex_from" 
      

    let index (s:string) (c:char) =  index_from s 0 c
    let rindex (s:string) (c:char) =  rindex_from s (length s - 1) c

    let contains_between (s:string) (start:int) (stop:int) (c:char) =  
      try s.IndexOf(c,start,(stop-start+1)) <> -1
      with :? System.ArgumentException -> invalid_arg "String.contains_between" 
        

    let contains_from (s:string) (start:int) (c:char) =  
      let stop = length s - 1 in 
      try s.IndexOf(c,start,(stop-start+1)) <> -1
      with :? System.ArgumentException -> invalid_arg "String.contains_from" 

    let rcontains_from (s:string) (stop:int) (c:char) =  
      let start = 0 in
      try s.IndexOf(c,start,(stop-start+1)) <> -1
      with :? System.ArgumentException -> invalid_arg "String.rcontains_from" 
      
    let contains (s:string) (c:char) =  contains_from s 0 c

    let uppercase (s:string) =  s.ToUpper()
    let lowercase (s:string) =  s.ToLower()

    let capitalize (s:string) =  
      if length s = 0 then invalid_arg "String.capitalize";
      concat "" [uppercase (sub s 0 1); sub s 1 (length s - 1)]

    let uncapitalize (s:string) =  
      if length s = 0 then invalid_arg "String.uncapitalize";
      concat "" [lowercase (sub s 0 1); sub s 1 (length s - 1)]

    #if CLI_AT_MOST_1_1
    #else
    let split (c : char list) = 
      let ca = CompatArray.of_list c in 
      fun (s:string) -> CompatArray.to_list(s.Split(ca, System.StringSplitOptions.RemoveEmptyEntries))
      
    let trim (c : char list) = 
      let ca = CompatArray.of_list c in 
      fun (s:string) -> s.Trim(ca)
    #endif
