/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store;

import java.io.PrintStream;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Map;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ManagedConnection;
import org.datanucleus.OMFContext;
import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.api.ApiAdapter;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusOptimisticException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.identity.OID;
import org.datanucleus.identity.SCOID;
import org.datanucleus.management.runtime.StoreManagerRuntime;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.IdentityStrategy;
import org.datanucleus.metadata.SequenceMetaData;
import org.datanucleus.metadata.VersionMetaData;
import org.datanucleus.sco.IncompatibleFieldTypeException;
import org.datanucleus.store.schema.StoreSchemaHandler;
import org.datanucleus.store.scostore.Store;
import org.datanucleus.store.valuegenerator.ValueGenerationManager;

/**
 * Interface defining management of a datastore.
 * To be implemented by all new datastore support. Please use AbstractStoreManager and extend it.
 */
public interface StoreManager
{
    /**
     * Accessor for the supported options in string form
     */
    public Collection getSupportedOptions();

    /**
     * Release of resources.
     */
    public void close();

    /**
     * Accessor for the runtime management of this component (if specified as required to the PMF/EMF).
     * @return Runtime manager (JMX) for this store
     */
    public StoreManagerRuntime getRuntimeManager();

    /**
     * Accessor for the store persistence handler.
     * @return Store persistence handler.
     */
    public StorePersistenceHandler getPersistenceHandler();

    /**
     * Accessor for the store schema handler (if this datastore supports the concept of a schema).
     * @return Store schema handler.
     */
    public StoreSchemaHandler getSchemaHandler();

    /**
     * Method to return a datastore sequence for this datastore matching the passed sequence MetaData.
     * @param om The Object Manager
     * @param seqmd SequenceMetaData
     * @return The Sequence
     */
    public NucleusSequence getNucleusSequence(ObjectManager om, SequenceMetaData seqmd);

    /**
     * Method to return a connection for the ObjectManager.
     * @param om ObjectManager
     * @return The JPOX Connection
     */
    public NucleusConnection getNucleusConnection(ObjectManager om);

    /**
     * Accessor for a connection for the specified ObjectManager.<p>
     * If there is an active transaction, a connection from the transactional
     * connection factory will be returned. If there is no active transaction,
     * a connection from the nontransactional connection factory will be returned.
     * @param om The ObjectManager
     * @return The Connection
     * @throws NucleusException Thrown if an error occurs getting the connection
     */
    public ManagedConnection getConnection(ObjectManager om);

    /**
     * Accessor for a connection for the specified ObjectManager.<p>
     * If there is an active transaction, a connection from the transactional
     * connection factory will be returned. If there is no active transaction,
     * a connection from the nontransactional connection factory will be returned.
     * @param om The ObjectManager
     * @param options connetion options
     * @return The Connection
     * @throws NucleusException Thrown if an error occurs getting the connection
     */
    public ManagedConnection getConnection(ObjectManager om, Map options);
    
    /**
     * Accessor for the ValueGenerationManager for obtaining sequences.
     * @return The ValueGenerationManagerr for this datastore
     */
    public ValueGenerationManager getValueGenerationManager();

    /**
     * Accessor for the API adapter.
     * @return API adapter
     */
    public ApiAdapter getApiAdapter();

    /**
     * Accessor for the key for this store manager.
     * @return StoreManager key
     */
    public String getStoreManagerKey();

    /**
     * Accessor for the context in which this RDBMSManager is running
     * @return Returns the context.
     */
    public OMFContext getOMFContext();

    /**
     * Get the date/time of the datastore.
     * @return Date/time of the datastore
     */
    public Date getDatastoreDate();

    /**
     * Method to output particular information owned by this datastore.
     * Each StoreManager can support whichever categories it likes.
     * @param cat Category of information 
     * @param ps PrintStream
     * @throws Exception Thrown if an error occurs in the output process
     */
    public void printInformation(String cat, PrintStream ps) throws Exception;

    /**
     * Accessor for the {@link AutoStartMechanism}
     * @return may return null if auto start mechanism is not initialized
     */
    public AutoStartMechanism getAutoStartMechanism();

    /**
     * Accessor for whether the specified class is managed currently
     * @param className The name of the class
     * @return Whether it is managed
     */
    public boolean managesClass(String className);

    /**
     * Method to add a class to the managed list for this datastore manager.
     * @param className Name of the class
     * @param clr The ClassLoaderResolver
     */
    public void addClass(String className, ClassLoaderResolver clr);

    /**
     * Add classes to the persistence model for the datastore.
     * <p>
     * This method is primarily useful for applications that wish to perform all
     * of their datastore initialization up front, rather than wait for the runtime to do it on-demand.
     * @param classNames The class(es) to be added.
     * @param clr The ClassLoaderResolver
     * @exception org.datanucleus.store.exceptions.DatastoreValidationException
     *      If there is some mismatch between the current datastore contents and
     *      those necessary to enable persistence of the given classes.
     */
    public void addClasses(String[] classNames, ClassLoaderResolver clr);

    /**
     * Remove all classes from the persistence model for the datastore.
     * This empties the datastore of all datastore objects managed by us.
     * All objects of types not managed are left untouched.
     * In the case of RDBMS this means drop all tables for types managed by us.
     * @param clr The ClassLoaderResolver
     */
    public void removeAllClasses(ClassLoaderResolver clr);

    /**
     * Convenience method to ensure that the class defined by the passed OID/SingleFIeldIdentity is 
     * managed by the store.
     * @param id OID
     * @param clr ClassLoader resolver
     * @return The class name of the class associated to this identity
     * @throws NucleusUserException if the identity is assigned to the wrong class
     */
    public String manageClassForIdentity(Object id, ClassLoaderResolver clr);

    /**
     * Interface to getting an Extent for a class.
     * @param om The Object Manager
     * @param c  The class requiring the Extent
     * @param subclasses Whether to include subclasses of 'c'
     * @return The Extent.
     */
    public Extent getExtent(ObjectManager om, Class c, boolean subclasses);

    /**
     * Accessor for whether this query language is supported.
     * @param language The language
     * @return Whether it is supported.
     */
    public boolean supportsQueryLanguage(String language);

    /**
     * Accessor for whether this value strategy is supported.
     * @param strategy The strategy
     * @return Whether it is supported.
     */
    public boolean supportsValueStrategy(String strategy);

    /**
     * Returns the class corresponding to the given object identity. 
     * If the object is an OID (datastore-identity), return the PC class specified in the identity.
     * If the object is SingleFieldIdentity, return the PC class specified in the identity
     * If the object is an AppID PK, return the PC class that uses it.
     * If the object is a SCOID, return the SCO class. 
     * If the object is a PersistenceCapable class, return the class. 
     * @param id The identity of some object.
     * @param clr ClassLoader resolver
     * @param om Object Manager
     * @return For datastore identity, return the class of the corresponding
     * object. For application identity, return the class of the corresponding
     * object or null if object does not exist.
     * @exception ClassCastException If the type of ID is not recognized ({@link OID}
     * or {@link SCOID}).
     */
    public String getClassNameForObjectID(Object id, ClassLoaderResolver clr, ObjectManager om);

    /**
     * Check if the strategy is attributed by the database when the
     * PersistenceCapable object is inserted into the database
     * @param identityStrategy the identityStrategy
     * @param datastoreIdentityField Whether this is for the surrogate datastore identity field
     * @return if the object for the strategy is attributed by the database
     */
    public boolean isStrategyDatastoreAttributed(IdentityStrategy identityStrategy, boolean datastoreIdentityField);

    /**
     * Method to retrieve the value for a strategy for a particular field.
     * @param om The Object Manager
     * @param cmd AbstractClassMetaData for the class
     * @param absoluteFieldNumber The field number
     * @return The value
     */
    public Object getStrategyValue(ObjectManager om, AbstractClassMetaData cmd, int absoluteFieldNumber);

    /**
     * Perform an optimistic version check on the passed object, against the passed version in the datastore.
     * @param sm StateManager of the object to check
     * @param versionDatastore Version of the object in the datastore
     * @param versionMetaData VersionMetaData to use for checking
     * @throws NucleusUserException thrown when an invalid strategy is specified
     * @throws NucleusOptimisticException thrown when the version check fails
     */
    public void performVersionCheck(StateManager sm, Object versionDatastore, VersionMetaData versionMetaData);

    /**
     * Utility to return the names of the classes that are known subclasses of the provided
     * class. Actually uses the MetaDataManager for determining what is a subclass
     * since the MetaData is often registered before being needed by the Store.
     * @param className Class for which we search for subclasses.
     * @param includeDescendents Whether to include subclasses of subclasses etc
     * @param clr The ClassLoaderResolver
     * @return Set of classes that are subclasses of the input class.
     * TODO Use method in MetaDataManager and remove this
     */
    public HashSet getSubClassesForClass(String className, boolean includeDescendents, ClassLoaderResolver clr);

    /**
     * Notifies this store manager that the main memory (RAM, heap) copy of the PC object  of the supplied
     * StateManager may not be regarded as valid anymore.
     * (The most common case is that the state of the PC becomes HOLLOW).
     * This is especially important for object databases employing implicit storing
     * from the main memory to the database (like DB4O).
     * These databases may stop tracking the main memory copy and linking it with its on-disk copy,
     * thus releasing memory.
     * More importantly, these databases then know that the object should be reloaded when it
     * is (maybe even implicitly) accessed again.
     *
     * To be clear: There may be multiple copies of the data of one PC object (from the user perspective),
     * namely a copy in main memory (on the Java heap) and a copy in the database (usually on disk).
     * As there may be multiple copies, some of these copies may be outdated or invalid. In case such
     * a copy is to be accessed, its contents should not be used. Rather than that, the outdated copy should
     * be overwritten by an authorative copy.
     *
     * This method marks the main memory copy of the object (on the Java heap) to be outdated in that sense.
     */
    public void notifyObjectIsOutdated(StateManager sm);

    /**
     * Accessor for the backing store for the specified field.
     * Override it if the store needs this
     * @param clr ClassLoader resolver
     * @param fmd MetaData for the field/property
     * @param type Type
     * @return Backing store
     * @throws IncompatibleFieldTypeException raises the exception if the field is not compatible if the store
     * @throws UnsupportedOperationException if BackingStore is not supported
     */
    public Store getBackingStoreForField(ClassLoaderResolver clr, AbstractMemberMetaData fmd, Class type);
}