/**********************************************************************
Copyright (c) 2002 Kelly Grizzle and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2002 Mike Martin - unknown changes
2003 Andy Jefferson - commented
2003 Andy Jefferson - added localiser
    ...
**********************************************************************/
package org.datanucleus.jpa.state;

import org.datanucleus.StateManager;
import org.datanucleus.Transaction;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.state.LifeCycleState;

/**
 * Class representing the life cycle state of PersistentDeleted.
 */
class PersistentDeleted extends LifeCycleState
{
    /** Protected Constructor to prevent external instantiation. */
    protected PersistentDeleted()
    {
		isPersistent = true;    	
        isDirty = true;
        isNew = false;
        isDeleted = true;
        isTransactional = true;

        stateType =  P_DELETED;
    }

    /**
     * Method to transition to non-transactional.
     * @param sm StateManager.
     * @return new LifeCycle state.
     */
    public LifeCycleState transitionMakeNontransactional(StateManager sm)
    {
        throw new NucleusUserException(LOCALISER.msg("027007"),sm.getInternalObjectId());
    }

    /**
     * Method to transition to transient.
     * @param sm StateManager.
     * @param useFetchPlan to make transient the fields in the fetch plan
     * @return new LifeCycle state.
     */
    public LifeCycleState transitionMakeTransient(StateManager sm, boolean useFetchPlan, boolean detachAllOnCommit)
    {
        throw new NucleusUserException(LOCALISER.msg("027008"),sm.getInternalObjectId());
    }

    /**
     * Method to transition to persistent state.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionMakePersistent(StateManager sm)
    {
        return changeState(sm, P_CLEAN);
    }

    /**
     * Method to transition to commit state.
     * @param sm StateManager.
     * @param tx the Transaction been committed.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionCommit(StateManager sm, Transaction tx)
    {
        sm.clearFields();
        return changeState(sm, TRANSIENT);
    }

    /**
     * Method to transition to rollback state.
     * @param sm StateManager.
     * @param tx The transaction
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionRollback(StateManager sm, Transaction tx)
    {
        if (tx.getRetainValues())
        {
            if (tx.getRestoreValues())
            {
                sm.restoreFields();
            }

            return changeState(sm, P_NONTRANS);
        }
        else
        {
            sm.clearNonPrimaryKeyFields();
            sm.clearSavedFields();
            return changeState(sm, HOLLOW);
        }

    }

    /**
     * Method to transition to write-field state.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionWriteField(StateManager sm)
    {
        throw new NucleusUserException(LOCALISER.msg("027010"),sm.getInternalObjectId());
    }

    /**
     * Method to return a string version of this object.
     * @return The string "P_DELETED".
     **/
    public String toString()
    {
        return "P_DELETED";
    }
}